#Requires -Module ActiveDirectory
#Requires -RunAsAdministrator
#Requires -Version 4

<#PSScriptInfo
.\ad.ps1
.\ad.ps1 -$Option1
.\ad.ps1 -$Option2
.\ad.ps1 -$Option3
#>

Param(
$reportpath = "$env:userprofile\desktop\ad",
[switch]$Option1,
[switch]$Option2,
[switch]$Option3
)

If (!($(Try { Test-Path $reportpath} Catch {$true}))){new-Item $reportpath -ItemType "directory"  -force}
cls
cd $reportpath
$searchbase = @()
$default_err_log = "$reportpath\ad.log"
$time_log = "$reportpath\ad.csv"
$forest_sids = ((get-adforest).domains | foreach{get-addomain -Server $_}).DomainSID.value

Function runReportArchival{
 param($Achivetokeep=5)
  #count the number of zip files in the directory and delete older zip files
  $achivecount = (Get-ChildItem $archivereportpath *.zip).count - $Achivetokeep
  if($achivecount -gt 0){
   Get-ChildItem $reportpath *.zip  | Sort CreationTime | Select -first $achivecount | Remove-Item -force
  }
  #archive reports
  if(get-command Compress-Archive){
   Write-host "Zipping up the reports"
   gci $reportpath\*.csv | Compress-Archive -DestinationPath "$reportpath\ADReport_$((Get-Date).ToString('MM-dd-yyyy_hh-mm-ss')).zip"
  }  
}
Function runallCollectfunctions{
 (dir function: | where name -like collectAD*).name | foreach{$script_function = "$($_) | export-csv $reportpath\$($_ -Replace("collect","report")).csv -notypeinformation"
  $runtime = Measure-Command {Invoke-Expression -Command $script_function} | `
   select @{name='RunDate';expression={get-date -format d}},`
   @{name='Function';expression={$script_function}}, `
   @{name='Hours';expression={$_.hours}}, `
   @{name='Minutes';expression={$_.Minutes}}, `
   @{name='Seconds';expression={$_.Seconds}}
   $runtime | export-csv $time_log -append -notypeinformation
 }
}
Function runSiteMappingfunctions{
 CollectADDomainControllers | export-csv $reportpath\reportADDomainControllers.csv -notypeinformation
 CollectADUsers -sitemap | export-csv $reportpath\reportADUsers.csv -notypeinformation
 CollectADComputers -sitemap | export-csv $reportpath\reportADComputers.csv -notypeinformation
 CollectADSiteDetails | export-csv $reportpath\reportADSiteDetails.csv -notypeinformation
 CollectADSubnets | export-csv $reportpath\reportADSubnets.csv -notypeinformation
 CollectADSiteLinks | export-csv $reportpath\reportADSiteLinks.csv -notypeinformation
}
function CleanupReports{
 param([switch]$runusercleanup,
 [switch]$rungroupcleanup,
 [switch]$runcomputercleanup,
 [switch]$runallcleanup)

 if($runusercleanup -or $runallcleanup){
  $results = @()
  #Create User Reports
  write-host "Generating Cleanup User Reports"
  $reportfile = "$reportpath\reportADUsers.csv"
  If ($(Try {Test-Path $reportfile} Catch {$false}))
  {
   write-host "Existing User Reports Found"
   $results = import-csv $reportfile
   $results | where sidhistory -eq "Review" | export-csv "$reportpath\cleanupADUsers-SidHistorySameDomain.csv" -NoTypeInformation
   $results | where DoesNotRequirePreAuth -eq $true | export-csv "$reportpath\cleanupADUsers-DoesNotRequirePreAuth.csv" -NoTypeInformation
   $results | where ReversibleEncryption -eq $true | export-csv "$reportpath\cleanupADUsers-ReversibleEncryption.csv" -NoTypeInformation
   $results | where UseDesKeyOnly -eq $true | export-csv "$reportpath\cleanupADUsers-UseDesKeyOnly.csv" -NoTypeInformation
   $results | where Trustedfordelegation -eq $true | export-csv "$reportpath\cleanupADUsers-UnConstrainedKerbDelegationEnabled.csv" -NoTypeInformation
   $results | where TrustedToAuthForDelegation -eq $true | export-csv "$reportpath\cleanupADUsers-KerbDelegationTransitioningEnabled.csv" -NoTypeInformation
   $results | where PasswordNeverSet -eq $true | export-csv "$reportpath\cleanupADUsers-PasswordNeverSet.csv" -NoTypeInformation
   $results | where PasswordNotRequired -eq $true | export-csv "$reportpath\cleanupADUsers-PasswordNotRequired.csv" -NoTypeInformation
   $results | where PasswordNeverExpires -eq $true | export-csv "$reportpath\cleanupADUsers-PasswordNeverExpires.csv" -NoTypeInformation
   $results | where Stale -eq $true | export-csv "$reportpath\cleanupADUsers-Stale.csv" -NoTypeInformation
   $results | where DefaultPrimaryGroup -ne $true | export-csv "$reportpath\cleanupADUsers-PrimaryGroupIDNotDomainUsers.csv" -NoTypeInformation
   #$results | where {$_.SPN -eq $true -and $_.admincount -eq 1} | export-csv "$reportpath\cleanupADUsers-PrimaryGroupIDNotDomainUsers.csv" -NoTypeInformation
   $results | where {[convert]::ToInt32($($_.PwdAgeinDays)) -gt 365} -ErrorAction SilentlyContinue | export-csv "$reportpath\cleanupADUsers-PWDAgeover1Year.csv" -NoTypeInformation
   $results | where {[convert]::ToInt32($($_.PwdAgeinDays)) -gt 1825} | export-csv "$reportpath\cleanupADUsers-PWDAgeover5Year.csv" -NoTypeInformation
   $results | where {[convert]::ToInt32($($_.PwdAgeinDays)) -gt 3650} | export-csv "$reportpath\cleanupADUsers-PWDAgeover10Year.csv" -NoTypeInformation
  }else{
   write-host "Existing User Reports Not Found"
   CollectADUsers | export-csv $reportfile -NoTypeInformation
   CleanupReports -runusercleanup
  }
 }
 if($rungroupcleanup -or $runallcleanup){
  $results = @()
  #Create Group Reports
  write-host "Generating Cleanup Group Reports"
  $reportfile = "$reportpath\reportADGroups.csv"
  If ($(Try {Test-Path $reportfile} Catch {$false}))
  {
   write-host "Existing Group Reports Found"
  }else{
   write-host "Existing Group Reports Not Found"
   CollectADGroups | export-csv $reportfile -NoTypeInformation
   CleanupReports -rungroupcleanup
  }
 }

 if($runcomputercleanup -or $runallcleanup){
  $results = @()
  #Create Group Reports
  write-host "Generating Cleanup Computer Reports"
  $reportfile = "$reportpath\reportADComputers.csv"
  If ($(Try {Test-Path $reportfile} Catch {$false}))
  {
   write-host "Existing Computer Reports Found"
  }else{
   write-host "Existing Computer Reports Not Found"
   CollectADComputers | export-csv $reportfile -NoTypeInformation
   CleanupReports -runcomputercleanup
  }
 }
}
Function showMenu{
 cls
 Write-host -ForegroundColor yellow "Select the AD Report to run:"
 Write-host "   0 - Collect All AD Reports"
 Write-host "   1 - Collect AD Computers"
 Write-host "   2 - Collect AD Groups"
 Write-host "   3 - Collect AD Users"
 Write-host "   4 - Collect AD Site Mapping Data"
 Write-host "   5 - Not Available"
 write-host "   6 - Collect AD Privileged Group History"
 Write-host "   7 - Not Available"
 Write-host "   8 - Not Available"
 Write-host "   9 - Not Available"
 Write-host "   10 - Create Cleanup List from Reports"
 $xMenuChoiceA = read-host "Please enter an option 0 to 10..."

 switch($xMenuChoiceA){
  0{runallCollectfunctions}
  1{#CollectADComputers
  $runtime = Measure-Command {CollectADComputers | export-csv $reportpath\reportADComputers.csv -notypeinformation} | `
    select @{name='RunDate';expression={get-date -format d}},`
    @{name='Function';expression={"CollectADComputers"}}, `
    @{name='Hours';expression={$_.hours}}, `
    @{name='Minutes';expression={$_.Minutes}}, `
    @{name='Seconds';expression={$_.Seconds}}
    $runtime | export-csv $time_log -append -notypeinfor6mation}
  2{#CollectADGroups
  $runtime = Measure-Command {CollectADGroups | export-csv $reportpath\reportADGroups.csv -notypeinformation} | `
    select @{name='RunDate';expression={get-date -format d}},`
    @{name='Function';expression={"CollectADGroups"}}, `
    @{name='Hours';expression={$_.hours}}, `
    @{name='Minutes';expression={$_.Minutes}}, `
    @{name='Seconds';expression={$_.Seconds}} 
    $runtime | export-csv $time_log -append -notypeinformation}
  3{#CollectADUsers
  $runtime = Measure-Command {CollectADUsers | export-csv $reportpath\reportADUsers.csv -notypeinformation} | `
    select @{name='RunDate';expression={get-date -format d}},`
    @{name='Function';expression={"CollectADUsers"}}, `
    @{name='Hours';expression={$_.hours}}, `
    @{name='Minutes';expression={$_.Minutes}}, `
    @{name='Seconds';expression={$_.Seconds}} 
    $runtime|export-csv $time_log -append -notypeinformation}
  4{runSiteMappingfunctions}
  5{}
  6{#CollectadPrivilegedGroupChanges
  $runtime = Measure-Command {CollectADPrivilegedGroupChanges | export-csv $reportpath\reportPrivilegedGroupChanges.csv -notypeinformation} | `
    select @{name='RunDate';expression={get-date -format d}},`
    @{name='Function';expression={"CollectADPrivilegedGroupChanges"}}, `
    @{name='Hours';expression={$_.hours}}, `
    @{name='Minutes';expression={$_.Minutes}}, `
    @{name='Seconds';expression={$_.Seconds}} 
    $runtime|export-csv $time_log -append -notypeinformation}
  10{CleanupReports -runallcleanup}
 }
}
#region Support Functions
function Get-ipSite{
	param([string]$ip
	)
	#Great idea from http://superuser.com/questions/758372/query-site-for-given-ip-from-ad-sites-and-services/758398
	$site = nltest /DSADDRESSTOSITE:$ip /dsgetsite 2>$null
	if ($LASTEXITCODE -eq 0) {
		$split = $site[3] -split "\s+"
		# validate result is for an IPv4 address before continuing
		if ($split[1] -match [regex]"^(?:(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.){3}(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)$") {
			"" | select @{l="ADSite";e={$split[2]}}, @{l="ADSubnet";e={$split[3]}}
		}
	}
}
#endregion
#region hashes
 $hash_domain = @{name='Domain';expression={$domain}}
 $hash_whenchanged = @{Name="whenchanged";
  Expression={($_.whenchanged).ToString('MM/dd/yyyy')}}
 $hash_whencreated = @{Name="whencreated";
  Expression={($_.whencreated).ToString('MM/dd/yyyy')}}
 $hash_NamingContext = @{name='NamingContext';
  expression={$sb}}
 $hash_pwdLastSet = @{Name="pwdLastSet";
  Expression={if($_.PwdLastSet -ne 0){([datetime]::FromFileTime($_.pwdLastSet).ToString('MM/dd/yyyy'))}}}
 $hash_lastLogonTimestamp = @{Name="LastLogonTimeStamp";
  Expression={if($_.LastLogonTimeStamp -like "*"){([datetime]::FromFileTime($_.LastLogonTimeStamp).ToString('MM/dd/yyyy'))}}}
 $hash_parentou = @{name='ParentOU';expression={
  $($_.distinguishedname -split '(?<![\\]),')[1..$($($_.distinguishedname -split '(?<![\\]),').Count-1)] -join ','}} 
 $hash_computerstale = @{Name="Stale";Expression={if(((($_.pwdLastSet -lt $d.ToFileTimeUTC()) -or ($_.pwdLastSet -ne 0)) -and 
  (($_.LastLogonTimeStamp -lt $d.ToFileTimeUTC()) -or ($_.LastLogonTimeStamp -notlike "*")) -and ($_.IPv4Address -eq $null)) -and 
  (!($_.serviceprincipalname -like "*MSClusterVirtualServer*"))){$True}else{$False}}}
 $hash_sidhistory = @{Name="sidHistory";Expression={if($_.sidhistory){if($forest_sids -like $_.SIDHistory) 
  {"Review"}else{$True}}else{$False}}}
 $hash_spn = @{Name="SPN";
  Expression={if($_.servicePrincipalName){$True}else{$False}}}
 $hash_PrimaryGroup = @{Name="DefaultPrimaryGroup";
  Expression={if($_.PrimaryGroupID -eq $Default_Group_ID){$True}else{$_.PrimaryGroupID}}}
 $hash_memlastchange = @{name='MembershipLastChanged';
  expression={if($_."msDS-ReplValueMetaData"){($_ | Select-Object -ExpandProperty "msDS-ReplValueMetaData" | 
   foreach {([XML]$_.Replace("`0","")).DS_REPL_VALUE_META_DATA | where { $_.pszAttributeName -eq "member" }} | 
   select -first 1).ftimeLastOriginatingChange | get-date -Format MM/dd/yyyy}else{"Never Used"}}}
 $hash_rid = @{name='Rid';expression={$([int]($_.objectsid -split("-"))[7])}}
 $hash_members = @{name='Member';
  expression={if($_.Member){$true}}}
 $hash_memberof = @{name='Memberof';expression={if($_.Memberof){$true}}}
 $hash_oudirectChild = @{name='DirectChildCount';
  expression={$_."msds-approx-immed-subordinates"}}
 $hash_gpLink = @{name='Linked GPOs';expression={if($_.gplink){$true}}}
 $hash_gpOptions = @{name='GPO Inheritance';expression={if($_.gpOptions -eq 1){"Blocked"}}}
 $hash_FGPPAppliesTo = @{name='AppliesToSet';expression={if($_.AppliesTo){$true}}}
 $hash_AccountNotDelegated = @{Name="AccountNotDelegated";
  Expression={if($_.UserAccountControl -band 1048576){$True}else{$False}}}
 $hash_UseDesKeyOnly = @{Name="UseDesKeyOnly";Expression={if($_.UserAccountControl -band 2097152){$True}else{$False}}}
 $hash_ReversibleEncryption = @{Name="ReversibleEncryption";
  Expression={if($_.UserAccountControl -band 128){$True}else{$False}}}
 $hash_DoesNotRequirePreAuth = @{Name="DoesNotRequirePreAuth";
  Expression={if($_.UserAccountControl -band 4194304){$True}else{$False}}}
 $hash_PasswordNeverExpires = @{Name="PasswordNeverExpires";
  Expression={if($_.UserAccountControl -band 65536){$True}else{$False}}}
 $hash_PasswordNotRequired = @{Name="PasswordNotRequired";
  Expression={if($_.UserAccountControl -band 32){$True}else{$False}}}
 $hash_PasswordNeverSet = @{Name="PasswordNeverSet";
  Expression={if($_.pwdLastSet -eq 0){$True}else{$False}}}
 $hash_SmartCardRequired = @{Name="SmartCardRequired";
  Expression={if($_.UserAccountControl -band 262144){$True}else{$False}}}
 $hash_AuthNPolicy = @{Name="AuthNPolicy";Expression={if("msDS-AssignedAuthNPolicy"){$True}else{$False}}}
 $hash_AuthNPolicySilo = @{Name="AuthNPolicySilo";
  Expression={if("msDS-AssignedAuthNPolicySilo"){$True}else{$False}}}
 $hash_PwdAgeinDays = @{Name="PwdAgeinDays";
  Expression={if($_.PwdLastSet -ne 0){(new-TimeSpan([datetime]::FromFileTimeUTC($_.PwdLastSet)) $(Get-Date)).days}else{0}}}
 $hash_userstale = @{Name="Stale";Expression={if((($_.pwdLastSet -lt $d.ToFileTimeUTC()) -or ($_.pwdLastSet -ne 0)) -and 
  (($_.LastLogonTimeStamp -lt $d.ToFileTimeUTC()) -or ($_.LastLogonTimeStamp -notlike "*")) -and 
  ($_.whencreated -lt $d)){$True}else{$False}}}
 $hash_istgOrphaned = @{name='istgOrphaned';expression={if($_.interSiteTopologyGenerator -like "*0ADEL*"){$true}else{$false}}}
 $hash_subnetcount = @{name='SubnetCount';expression={($_.siteObjectBL | measure-object).count}}
 $hash_dcinsitecount = @{name='DCinSiteCount';expression={(Get-ADObject -Filter 'objectClass -eq "server"'`
      -searchbase $_.DistinguishedName | measure-object).count}}
 $hash_sitelinkcount = @{name='SiteLinkCount';expression={$sn = $(($_).name);
  (Get-ADReplicationSiteLink -Filter 'SitesIncluded -eq $sn' | measure-object).count}}
 $hash_SiteAddress = @{name='Address';expression={}}
 $hash_SiteCity =@{name='City';expression={}}
 $hash_SiteState = @{name='State';expression={}}
 $hash_SiteCountry = @{name='Country';expression={}}
 $hash_SubnetSiteName = @{name='SiteName';expression={if($_.siteobject){(Get-ADReplicationSite $_.siteObject).name}else{$false}}}
 $hash_SiteCount = @{name='SiteCount';expression={($_.siteList | measure).count}}
 $hash_SiteString = @{name='Sites';expression={$SiteList}}
 $hash_usercity = @{name='City';expression={$_.l}}
 $hash_userstate = @{name='State';expression={$_.st}}
 $hash_userc = @{name='Country';expression={$_.c}} 
 $hash_userco = @{name='Country1';expression={$_.co}}
 $hash_usercountrycode = @{name='Country2';expression={$_.countryCode}}
#endregion

if(!($skipMenu)){
 showMenu
}else{
 runallCollectfunctions
}

#runs Function to create a compressed version of the script
runReportArchival

$results = $null
write-host "Report Can be found here $reportpath"

dir function: | where name -like collectAD* | remove-item -force

