$ScriptPath = $MyInvocation.MyCommand.Path
$SettingsPath = $ScriptPath -replace '\.ps1$', '.ini'
$ScriptLog = $ScriptPath -replace '\.ps1$', '.log'

if (-not (Test-Path $SettingsPath)) {
    $settingsMessage = @"
RefreshToken=
ClientID=
ClientSecret=
SourceFile=
ParentId=
MaxLogFileLines=1000
"@
    $settingsMessage | Out-File -FilePath $SettingsPath -Encoding UTF8
    Write-Host "The settings file '$SettingsPath' was created."
    Write-Host "Please update all the variables in the settings file before running the script."
}

if (-not (Test-Path $ScriptLog)) {
    $logHeader = 'Date,Time,Type,Message'
    $logHeader | Out-File -FilePath $ScriptLog -Encoding UTF8
    Write-Host "The log file '$ScriptLog' was created."
}

function Write-Log {
    param([string]$Type, [string]$Message)
    $logEntry = "$(Get-Date -Format 'yyyy-MM-dd'),$(Get-Date -Format 'HH:mm:ss'),$Type,$Message"
    $logEntry | Out-File -FilePath $ScriptLog -Encoding UTF8 -Append
    Write-Host $logEntry

    $logLines = Get-Content -Path $ScriptLog
    if ($logLines.Count -gt $MaxLogFileLines) {
        $logLines | Select-Object -Last $MaxLogFileLines | Set-Content -Path $ScriptLog
    }
}

$requiredSettings = @('RefreshToken', 'ClientID', 'ClientSecret', 'ParentId', 'MaxLogFileLines')
$missingSettings = $requiredSettings | Where-Object { -not $Settings.ContainsKey($_) }

if ($missingSettings) {
    Write-Log -Type 'ERROR' -Message "The following required settings are missing in the settings file:"
    $missingSettings | ForEach-Object { Write-Log -Type 'ERROR' -Message "- $_" }
    Write-Log -Type 'ERROR' -Message "Please update the missing settings in the settings file before running the script."
    return
}

$RefreshToken = $Settings['RefreshToken']
$ClientID = $Settings['ClientID']
$ClientSecret = $Settings['ClientSecret']
$ParentId = $Settings['ParentId']
$MaxLogFileLines = $Settings['MaxLogFileLines']

Write-Log -Type 'START' -Message 'Script started.'

foreach ($SourceFile in $Settings['SourceFile'] -split '\s*,\s*') {
    $SourceFileHash = "$SourceFile.hash"
    $previousHash = ''
    if (Test-Path $SourceFileHash) {
        $previousHash = Get-Content $SourceFileHash
    }

    $currentHash = Get-FileHash -Path $SourceFile | Select-Object -ExpandProperty Hash

    if ($currentHash -eq $previousHash) {
        Write-Log -Type 'INFO' -Message "File hash for '$SourceFile' is the same. No need to upload."
        continue
    } else {
        Write-Log -Type 'INFO' -Message "File hash for '$SourceFile' is different. Proceeding with the upload."
        Set-Content -Path $SourceFileHash -Value $currentHash
    }

    $accessToken = (Invoke-RestMethod @{
        Uri = 'https://accounts.google.com/o/oauth2/token'
        Body = "refresh_token=$RefreshToken&client_id=$ClientID&client_secret=$ClientSecret&grant_type=refresh_token"
        Method = 'Post'
        ContentType = 'application/x-www-form-urlencoded'
    }).access_token

    $sourceBase64 = [Convert]::ToBase64String([IO.File]::ReadAllBytes((Get-Item $SourceFile).FullName))
    $sourceMime = [System.Web.MimeMapping]::GetMimeMapping((Get-Item $SourceFile).FullName)
    $supportsTeamDrives = 'false'

    $uploadMetadata = @{
        originalFilename = (Get-Item $SourceFile).Name
        name = (Get-Item $SourceFile).Name
        parents = @($ParentId)
    }

    $uploadBody = @"
--boundary
Content-Type: application/json; charset=UTF-8

$($uploadMetadata | ConvertTo-Json)

--boundary
Content-Transfer-Encoding: base64
Content-Type: $sourceMime

$sourceBase64
--boundary--
"@

    $uploadHeaders = @{
        "Authorization" = "Bearer $accessToken"
        "Content-Type" = 'multipart/related; boundary=boundary'
        "Content-Length" = $uploadBody.Length
    }

    $response = Invoke-RestMethod -Uri "https://www.googleapis.com/upload/drive/v3/files?uploadType=multipart&supportsTeamDrives=$supportsTeamDrives" -Method Post -Headers $uploadHeaders -Body $uploadBody

    Write-Log -Type 'INFO' -Message "File '$SourceFile' uploaded."
}

Write-Log -Type 'STOP' -Message 'Script stopped.'

